Copyright (c) 2007 dhpoware. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

-------------------------------------------------------------------------------

Tangent space parallax normal mapping shader for a single spot light source.

We borrow the Direct3D convention of an inner and an outer cone for the spot
light. A direct per-fragment implementation of the OpenGL spot light model will
produce a very sharp falloff around the spotCutoff. Using the inner and outer
cone convention will produce a more softer falloff around the spotCutoff.

To enable the inner and outer cone version of the spot light fragment shader
define the preprocessor symbol USE_INNER_AND_OUTER_CONES. If this symbol is not
defined the fragment shader will use the OpenGL spot light model.

Parallax mapping is a method for approximating the correct appearance of uneven
surfaces by modifying the texture coordinate for each fragment of the surface.
This provides the illusion of depth to such surfaces.

The theory behind parallax mapping with offset limiting is explained here:
http://web.archive.org/web/20060207121301/http://www.infiscape.com/doc/parallax_mapping.pdf 

The vertex shader is a standard tangent space normal mapping vertex shader
where the light and view vectors are transformed into tangent space.

The fragment shader is almost the same as a standard tangent space normal
mapping fragment shader with the exception of the parallax mapping term. Rather
than using the interpolated texture coordinate from the vertex shader to index
the normal and color map textures a new texture coordinate is calculated and
used instead.

The attenuation is calculated using this formula:

	attenuation factor = 1 - ((d * d) / (r * r))

Where
	d = distance between the light's position and the vertex
	r = radius of the light
    
This formula simplifies to:

	lightDir = (gl_LightSource[0].position.xyz - position) / lightRadius
	attenuation factor = 1 - dot(lightDir, lightDir)

Finally the attenuation factor is clamped to prevent negative values:

	attenuation factor = max(0, 1 - dot(lightDir, lightDir))

-------------------------------------------------------------------------------

[vert]

#version 110

uniform float lightRadius;

varying vec3 lightDir;
varying vec3 spotDir;
varying vec3 viewDir;

void main()
{
    gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex;
    gl_TexCoord[0] = gl_MultiTexCoord0;

    vec3 vertexPos = vec3(gl_ModelViewMatrix * gl_Vertex);

    vec3 n = normalize(gl_NormalMatrix * gl_Normal);
    vec3 t = normalize(gl_NormalMatrix * gl_MultiTexCoord1.xyz);
    vec3 b = cross(n, t) * gl_MultiTexCoord1.w;
    
    mat3 tbnMatrix = mat3(t.x, b.x, n.x,
                          t.y, b.y, n.y,
                          t.z, b.z, n.z);
    
    lightDir = (gl_LightSource[0].position.xyz - vertexPos) / lightRadius;
    lightDir = tbnMatrix * lightDir;

    spotDir = gl_LightSource[0].spotDirection;
    spotDir = tbnMatrix * spotDir;

    viewDir = -vertexPos;
    viewDir = tbnMatrix * viewDir;
}

[frag]

#version 110

#define USE_INNER_AND_OUTER_CONES

uniform sampler2D colorMap;
uniform sampler2D normalMap;
uniform sampler2D heightMap;

#if defined(USE_INNER_AND_OUTER_CONES)
    uniform float cosOuterCone;
    uniform float cosInnerCone;
#endif

uniform bool enableParallax;
uniform float scale;
uniform float bias;

varying vec3 lightDir;
varying vec3 spotDir;
varying vec3 viewDir;

void main()
{
    vec2 newTexCoord;
    vec3 v = normalize(viewDir);

    if (enableParallax == true)
    {
        float height = texture2D(heightMap, gl_TexCoord[0].st).r;
        
        height = height * scale + bias;
        newTexCoord = gl_TexCoord[0].st + (height * v.xy);
    }
    else
    {
        newTexCoord = gl_TexCoord[0].st;
    }

    vec3 l = lightDir;
    float atten = max(0.0, 1.0 - dot(l, l));
    
    l = normalize(l);
    
    float spotDot = dot(-l, normalize(spotDir));

#if defined(USE_INNER_AND_OUTER_CONES)
    float spotEffect = smoothstep(cosOuterCone, cosInnerCone, spotDot);
#else
    float spotEffect = (spotDot < gl_LightSource[0].spotCosCutoff)
                        ? 0.0 : pow(spotDot, gl_LightSource[0].spotExponent);
#endif

    atten *= spotEffect;

    vec3 n = normalize(texture2D(normalMap, newTexCoord).rgb * 2.0 - 1.0);
    vec3 h = normalize(l + v);

    float nDotL = max(0.0, dot(n, l));
    float nDotH = max(0.0, dot(n, h));
    float power = (nDotL == 0.0) ? 0.0 : pow(nDotH, gl_FrontMaterial.shininess);
    
    vec4 ambient = gl_FrontLightProduct[0].ambient * atten;
    vec4 diffuse = gl_FrontLightProduct[0].diffuse * nDotL * atten;
    vec4 specular = gl_FrontLightProduct[0].specular * power * atten;
    vec4 color = gl_FrontLightModelProduct.sceneColor + ambient + diffuse + specular;
    
    gl_FragColor = color * texture2D(colorMap, newTexCoord);
}
