//-----------------------------------------------------------------------------
// Copyright (c) 2006-2008 dhpoware. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//-----------------------------------------------------------------------------
//
// This demo implements tangent space parallax normal mapping using a series of
// OpenGL Shading Language (GLSL) shader programs. Separate parallax normal
// mapping shader programs are provided for each of the three OpenGL light
// types: directional, point, and spot lights.
//
// This demo extends the normal mapping shader demo by adding parallax mapping
// with offset limiting. Download the normal mapping shader from
// http://www.dhpoware.com/downloads/GLSLNormalMapping.zip.
//
// Parallax mapping is a method for approximating the correct appearance of
// uneven surfaces by modifying the texture coordinate for each pixel. Normal
// mapping is commonly used to provide surfaces with very detailed shading
// without requiring additional geometry to be added to the polygonal model.
// However such normal mapped surfaces often still appear flat. Parallax
// mapping corrects this by providing the illusion of depth to such surfaces.
// Parallax mapping is commonly used in conjunction with normal mapping because
// parallax mapping is a relatively inexpensive means to enhance the normal
// mapping effect.
//
// The original idea for parallax mapping came from a paper by Tomomichi Kaneko
// et al (2001) titled "Detailed Shape Representation with Parallax Mapping".
// Later Terry Welsh (2004) introduced an offset limiting term to the original
// parallax mapping technique in his paper titled "Parallax Mapping with Offset
// Limiting: A PerPixel Approximation of Uneven Surfaces".
//
// Terry Welsh popularized parallax mapping. In 2004 he released a working
// implementation of this technique to the OpenGL developer community. Within
// days of posting his code onto the OpenGL.org forums many developers across
// the world had already added this technique to their own applications.
//
// Terry Welsh's original OpenGL.org forum posting can be found here:
// http://www.opengl.org/discussion_boards/ubb/Forum3/HTML/011292.html
//
// Terry Welsh's parallax mapping with offset limiting paper can be found here:
// http://web.archive.org/web/20060207121301/http://www.infiscape.com/doc/parallax_mapping.pdf 
//
// The following OpenGL extensions are used where available:
//  GL_ARB_multisample
//  GL_EXT_texture_filter_anisotropic
//  WGL_ARB_extensions_string
//  WGL_ARB_multisample
//  WGL_ARB_pixel_format
//  WGL_EXT_swap_control
//
//-----------------------------------------------------------------------------

#if !defined(WIN32_LEAN_AND_MEAN)
#define WIN32_LEAN_AND_MEAN
#endif

#include <windows.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <cmath>
#include <fstream>
#include <sstream>
#include <stdexcept>
#include <string>
#include <vector>

#if defined(_DEBUG)
#include <crtdbg.h>
#endif

#include "bitmap.h"
#include "gl_font.h"
#include "mathlib.h"
#include "opengl.h"
#include "WGL_ARB_multisample.h"

//-----------------------------------------------------------------------------
// Constants.
//-----------------------------------------------------------------------------

#define APP_TITLE "GLSL Parallax Normal Mapping Demo"

// Windows Vista compositing support.
#if !defined(PFD_SUPPORT_COMPOSITION)
#define PFD_SUPPORT_COMPOSITION 0x00008000
#endif

// GL_EXT_texture_filter_anisotropic
#define GL_TEXTURE_MAX_ANISOTROPY_EXT     0x84FE
#define GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT 0x84FF

const float CAMERA_FOVY = 45.0f;
const float CAMERA_ZFAR = 100.0f;
const float CAMERA_ZNEAR = 0.1f;

const float DOLLY_MAX = 10.0f;
const float DOLLY_MIN = 2.5f;

const float MOUSE_ORBIT_SPEED = 0.30f;
const float MOUSE_DOLLY_SPEED = 0.02f;
const float MOUSE_TRACK_SPEED = 0.005f;

const float SPOT_INNER_CONE = 10.0f;
const float SPOT_OUTER_CONE = 15.0f;

const float LIGHT_RADIUS = 10.0f;

//-----------------------------------------------------------------------------
// User Defined Types.
//-----------------------------------------------------------------------------

enum {DIR_LIGHT, POINT_LIGHT, SPOT_LIGHT};

struct Vertex
{
    float pos[3];
    float texcoord[2];
    float normal[3];
    float tangent[4];
};

//-----------------------------------------------------------------------------
// Globals.
//-----------------------------------------------------------------------------

HWND        g_hWnd;
HDC         g_hDC;
HGLRC       g_hRC;
HINSTANCE   g_hInstance;
int         g_framesPerSecond;
int         g_windowWidth;
int         g_windowHeight;
int         g_msaaSamples;
int         g_maxAnisotrophy;
int         g_lightType = DIR_LIGHT;
float       g_cosInnerCone;
float       g_cosOuterCone;
float       g_heading;
float       g_pitch;
float       g_eyePos[4] = {0.0f, 0.0f, 4.0f, 1.0f};
float       g_lightPos[4] = {0.0f, 0.0f, 1.0f, 0.0f};
float       g_targetPos[4] = {0.0f, 0.0f, 0.0f, 1.0f};
float       g_scaleBias[2] = {0.04f, -0.03f};
bool        g_isFullScreen;
bool        g_hasFocus;
bool        g_enableVerticalSync;
bool        g_wireframe;
bool        g_disableColorMapTexture;
bool        g_displayHelp;
bool        g_disableParallax;
GLuint      g_vertexBuffer;
GLuint      g_nullTexture;
GLuint      g_colorMapTexture;
GLuint      g_normalMapTexture;
GLuint      g_heightMapTexture;
GLuint      g_dirParallaxNormalMappingShaderProgram;
GLuint      g_pointParallaxNormalMappingShaderProgram;
GLuint      g_spotParallaxNormalMappingShaderProgram;
GLFont      g_font;

Vertex g_cube[24] =
{
    // Positive Z Face.
    { -1.0f, -1.0f,  1.0f,  0.0f, 0.0f,  0.0f, 0.0f, 1.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f, -1.0f,  1.0f,  1.0f, 0.0f,  0.0f, 0.0f, 1.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f,  1.0f,  1.0f,  1.0f, 1.0f,  0.0f, 0.0f, 1.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    { -1.0f,  1.0f,  1.0f,  0.0f, 1.0f,  0.0f, 0.0f, 1.0f,  0.0f, 0.0f, 0.0f, 0.0f},

    // Negative Z Face.
    {  1.0f, -1.0f, -1.0f,  0.0f, 0.0f,  0.0f, 0.0f, -1.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    { -1.0f, -1.0f, -1.0f,  1.0f, 0.0f,  0.0f, 0.0f, -1.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    { -1.0f,  1.0f, -1.0f,  1.0f, 1.0f,  0.0f, 0.0f, -1.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f,  1.0f, -1.0f,  0.0f, 1.0f,  0.0f, 0.0f, -1.0f,  0.0f, 0.0f, 0.0f, 0.0f},

    // Positive Y Face.
    { -1.0f,  1.0f,  1.0f,  0.0f, 0.0f,  0.0f, 1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f,  1.0f,  1.0f,  1.0f, 0.0f,  0.0f, 1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f,  1.0f, -1.0f,  1.0f, 1.0f,  0.0f, 1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    { -1.0f,  1.0f, -1.0f,  0.0f, 1.0f,  0.0f, 1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},

    // Negative Y Face.
    { -1.0f, -1.0f, -1.0f,  0.0f, 0.0f,  0.0f, -1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f, -1.0f, -1.0f,  1.0f, 0.0f,  0.0f, -1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f, -1.0f,  1.0f,  1.0f, 1.0f,  0.0f, -1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    { -1.0f, -1.0f,  1.0f,  0.0f, 1.0f,  0.0f, -1.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},

    // Positive X Face.
    {  1.0f, -1.0f,  1.0f,  0.0f, 0.0f,  1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f, -1.0f, -1.0f,  1.0f, 0.0f,  1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f,  1.0f, -1.0f,  1.0f, 1.0f,  1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    {  1.0f,  1.0f,  1.0f,  0.0f, 1.0f,  1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},

    // Negative X Face.
    { -1.0f, -1.0f, -1.0f,  0.0f, 0.0f,  -1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    { -1.0f, -1.0f,  1.0f,  1.0f, 0.0f,  -1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    { -1.0f,  1.0f,  1.0f,  1.0f, 1.0f,  -1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f},
    { -1.0f,  1.0f, -1.0f,  0.0f, 1.0f,  -1.0f, 0.0f, 0.0f,  0.0f, 0.0f, 0.0f, 0.0f}
};

//-----------------------------------------------------------------------------
// Functions Prototypes.
//-----------------------------------------------------------------------------

void    CalcTangentVector(const float pos1[3], const float pos2[3],
                          const float pos3[3], const float texCoord1[2],
                          const float texCoord2[2], const float texCoord3[2],
                          const float normal[3], float tangent[4]);
void    Cleanup();
void    CleanupApp();
GLuint  CompileShader(GLenum type, const GLchar *pszSource, GLint length);
HWND    CreateAppWindow(const WNDCLASSEX &wcl, const char *pszTitle);
GLuint  CreateNullTexture(int width, int height);
void    EnableVerticalSync(bool enableVerticalSync);
float   GetElapsedTimeInSeconds();
LPCSTR  GetLightTypeStr();
bool    Init();
void    InitApp();
void    InitCube();
void    InitGL();
GLuint  LinkShaders(GLuint vertShader, GLuint fragShader);
GLuint  LoadShaderProgram(const char *pszFilename, std::string &infoLog);
GLuint  LoadTexture(const char *pszFilename);
GLuint  LoadTexture(const char *pszFilename, GLint magFilter, GLint minFilter,
                    GLint wrapS, GLint wrapT);
void    Log(const char *pszMessage);
void    ProcessMouseInput(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
void    ReadTextFile(const char *pszFilename, std::string &buffer);
void    RenderCube();
void    RenderFrame();
void    RenderText();
void    SetProcessorAffinity();
void    ToggleFullScreen();
void    UpdateFrame(float elapsedTimeSec);
void    UpdateFrameRate(float elapsedTimeSec);
void    UpdateLighting();
LRESULT CALLBACK WindowProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);

//-----------------------------------------------------------------------------
// Functions.
//-----------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nShowCmd)
{
#if defined _DEBUG
    _CrtSetDbgFlag(_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF);
    _CrtSetReportMode(_CRT_ASSERT, _CRTDBG_MODE_FILE);
    _CrtSetReportFile(_CRT_ASSERT, _CRTDBG_FILE_STDERR);
#endif

    MSG msg = {0};
    WNDCLASSEX wcl = {0};

    wcl.cbSize = sizeof(wcl);
    wcl.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
    wcl.lpfnWndProc = WindowProc;
    wcl.cbClsExtra = 0;
    wcl.cbWndExtra = 0;
    wcl.hInstance = g_hInstance = hInstance;
    wcl.hIcon = LoadIcon(0, IDI_APPLICATION);
    wcl.hCursor = LoadCursor(0, IDC_ARROW);
    wcl.hbrBackground = 0;
    wcl.lpszMenuName = 0;
    wcl.lpszClassName = "GLWindowClass";
    wcl.hIconSm = 0;

    if (!RegisterClassEx(&wcl))
        return 0;

    g_hWnd = CreateAppWindow(wcl, APP_TITLE);

    if (g_hWnd)
    {
        SetProcessorAffinity();

        if (Init())
        {
            ShowWindow(g_hWnd, nShowCmd);
            UpdateWindow(g_hWnd);

            while (true)
            {
                while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE))
                {
                    if (msg.message == WM_QUIT)
                        break;

                    TranslateMessage(&msg);
                    DispatchMessage(&msg);
                }

                if (msg.message == WM_QUIT)
                    break;

                if (g_hasFocus)
                {
                    UpdateFrame(GetElapsedTimeInSeconds());
                    RenderFrame();
                    SwapBuffers(g_hDC);
                }
                else
                {
                    WaitMessage();
                }
            }
        }

        Cleanup();
        UnregisterClass(wcl.lpszClassName, hInstance);
    }

    return static_cast<int>(msg.wParam);
}

LRESULT CALLBACK WindowProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg)
    {
    case WM_ACTIVATE:
        switch (wParam)
        {
        default:
            break;

        case WA_ACTIVE:
        case WA_CLICKACTIVE:
            g_hasFocus = true;
            break;

        case WA_INACTIVE:
            if (g_isFullScreen)
                ShowWindow(hWnd, SW_MINIMIZE);
            g_hasFocus = false;
            break;
        }
        break;

    case WM_CHAR:
        switch (static_cast<int>(wParam))
        {
        case VK_ESCAPE:
            PostMessage(hWnd, WM_CLOSE, 0, 0);
            break;

        case VK_SPACE:
            if (++g_lightType > SPOT_LIGHT)
                g_lightType = DIR_LIGHT;

            UpdateLighting();
            break;

        case VK_BACK:
            if (--g_lightType < DIR_LIGHT)
                g_lightType = SPOT_LIGHT;

            UpdateLighting();
            break;

        case 'B':
            g_scaleBias[1] += 0.001f;
            break;

        case 'b':
            g_scaleBias[1] -= 0.001f;
            break;

        case 'H':
        case 'h':
            g_displayHelp = !g_displayHelp;
            break;
        
        case 'P':
        case 'p':
            g_disableParallax = !g_disableParallax;
            break;
        
        case 'S':
            g_scaleBias[0] += 0.01f;
            break;

        case 's':
            g_scaleBias[0] -= 0.01f;
            break;

        case 'T':
        case 't':
            g_disableColorMapTexture = !g_disableColorMapTexture;
            break;

        case 'V':
        case 'v':
            EnableVerticalSync(!g_enableVerticalSync);
            break;
        }
        break;

    case WM_DESTROY:
        PostQuitMessage(0);
        return 0;

    case WM_SIZE:
        g_windowWidth = static_cast<int>(LOWORD(lParam));
        g_windowHeight = static_cast<int>(HIWORD(lParam));
        break;

    case WM_SYSKEYDOWN:
        if (wParam == VK_RETURN)
            ToggleFullScreen();
        break;

    default:
        ProcessMouseInput(hWnd, msg, wParam, lParam);
        break;
    }

    return DefWindowProc(hWnd, msg, wParam, lParam);
}

void CalcTangentVector(const float pos1[3], const float pos2[3],
                       const float pos3[3], const float texCoord1[2],
                       const float texCoord2[2], const float texCoord3[2],
                       const float normal[3], float tangent[4])
{
    // Given the 3 vertices (position and texture coordinates) of a triangle
    // calculate and return the triangle's tangent vector.

    // Create 2 vectors in object space.
    //
    // edge1 is the vector from vertex positions pos1 to pos2.
    // edge2 is the vector from vertex positions pos1 to pos3.
    Vector3 edge1(pos2[0] - pos1[0], pos2[1] - pos1[1], pos2[2] - pos1[2]);
    Vector3 edge2(pos3[0] - pos1[0], pos3[1] - pos1[1], pos3[2] - pos1[2]);

    edge1.normalize();
    edge2.normalize();

    // Create 2 vectors in tangent (texture) space that point in the same
    // direction as edge1 and edge2 (in object space).
    //
    // texEdge1 is the vector from texture coordinates texCoord1 to texCoord2.
    // texEdge2 is the vector from texture coordinates texCoord1 to texCoord3.
    Vector2 texEdge1(texCoord2[0] - texCoord1[0], texCoord2[1] - texCoord1[1]);
    Vector2 texEdge2(texCoord3[0] - texCoord1[0], texCoord3[1] - texCoord1[1]);

    texEdge1.normalize();
    texEdge2.normalize();

    // These 2 sets of vectors form the following system of equations:
    //
    //  edge1 = (texEdge1.x * tangent) + (texEdge1.y * bitangent)
    //  edge2 = (texEdge2.x * tangent) + (texEdge2.y * bitangent)
    //
    // Using matrix notation this system looks like:
    //
    //  [ edge1 ]     [ texEdge1.x  texEdge1.y ]  [ tangent   ]
    //  [       ]  =  [                        ]  [           ]
    //  [ edge2 ]     [ texEdge2.x  texEdge2.y ]  [ bitangent ]
    //
    // The solution is:
    //
    //  [ tangent   ]        1     [ texEdge2.y  -texEdge1.y ]  [ edge1 ]
    //  [           ]  =  -------  [                         ]  [       ]
    //  [ bitangent ]      det A   [-texEdge2.x   texEdge1.x ]  [ edge2 ]
    //
    //  where:
    //        [ texEdge1.x  texEdge1.y ]
    //    A = [                        ]
    //        [ texEdge2.x  texEdge2.y ]
    //
    //    det A = (texEdge1.x * texEdge2.y) - (texEdge1.y * texEdge2.x)
    //
    // From this solution the tangent space basis vectors are:
    //
    //    tangent = (1 / det A) * ( texEdge2.y * edge1 - texEdge1.y * edge2)
    //  bitangent = (1 / det A) * (-texEdge2.x * edge1 + texEdge1.x * edge2)
    //     normal = cross(tangent, bitangent)

    Vector3 t;
    Vector3 b;
    Vector3 n(normal[0], normal[1], normal[2]);

    float det = (texEdge1.x * texEdge2.y) - (texEdge1.y * texEdge2.x);

    if (Math::closeEnough(det, 0.0f))
    {
        t.set(1.0f, 0.0f, 0.0f);
        b.set(0.0f, 1.0f, 0.0f);
    }
    else
    {
        det = 1.0f / det;

        t.x = (texEdge2.y * edge1.x - texEdge1.y * edge2.x) * det;
        t.y = (texEdge2.y * edge1.y - texEdge1.y * edge2.y) * det;
        t.z = (texEdge2.y * edge1.z - texEdge1.y * edge2.z) * det;

        b.x = (-texEdge2.x * edge1.x + texEdge1.x * edge2.x) * det;
        b.y = (-texEdge2.x * edge1.y + texEdge1.x * edge2.y) * det;
        b.z = (-texEdge2.x * edge1.z + texEdge1.x * edge2.z) * det;

        t.normalize();
        b.normalize();
    }

    // Calculate the handedness of the local tangent space.
    // The bitangent vector is the cross product between the triangle face
    // normal vector and the calculated tangent vector. The resulting bitangent
    // vector should be the same as the bitangent vector calculated from the
    // set of linear equations above. If they point in different directions
    // then we need to invert the cross product calculated bitangent vector. We
    // store this scalar multiplier in the tangent vector's 'w' component so
    // that the correct bitangent vector can be generated in the normal mapping
    // shader's vertex shader.

    Vector3 bitangent = Vector3::cross(n, t);
    float handedness = (Vector3::dot(bitangent, b) < 0.0f) ? -1.0f : 1.0f;

    tangent[0] = t.x;
    tangent[1] = t.y;
    tangent[2] = t.z;
    tangent[3] = handedness;
}

void Cleanup()
{
    CleanupApp();

    if (g_hDC)
    {
        if (g_hRC)
        {
            wglMakeCurrent(g_hDC, 0);
            wglDeleteContext(g_hRC);
            g_hRC = 0;
        }

        ReleaseDC(g_hWnd, g_hDC);
        g_hDC = 0;
    }
}

void CleanupApp()
{
    if (g_dirParallaxNormalMappingShaderProgram)
    {
        glDeleteProgram(g_dirParallaxNormalMappingShaderProgram);
        g_dirParallaxNormalMappingShaderProgram = 0;
    }

    if (g_pointParallaxNormalMappingShaderProgram)
    {
        glDeleteProgram(g_pointParallaxNormalMappingShaderProgram);
        g_pointParallaxNormalMappingShaderProgram = 0;
    }

    if (g_spotParallaxNormalMappingShaderProgram)
    {
        glDeleteProgram(g_spotParallaxNormalMappingShaderProgram);
        g_spotParallaxNormalMappingShaderProgram = 0;
    }

    if (g_nullTexture)
    {
        glDeleteTextures(1, &g_nullTexture);
        g_nullTexture = 0;
    }

    if (g_heightMapTexture)
    {
        glDeleteTextures(1, &g_heightMapTexture);
        g_heightMapTexture = 0;
    }

    if (g_normalMapTexture)
    {
        glDeleteTextures(1, &g_normalMapTexture);
        g_normalMapTexture = 0;
    }

    if (g_colorMapTexture)
    {
        glDeleteTextures(1, &g_colorMapTexture);
        g_colorMapTexture = 0;
    }

    if (g_vertexBuffer)
    {
        glDeleteBuffers(1, &g_vertexBuffer);
        g_vertexBuffer = 0;
    }

    g_font.destroy();
}

GLuint CompileShader(GLenum type, const GLchar *pszSource, GLint length)
{
    // Compiles the shader given it's source code. Returns the shader object.
    // A std::string object containing the shader's info log is thrown if the
    // shader failed to compile.
    //
    // 'type' is either GL_VERTEX_SHADER or GL_FRAGMENT_SHADER.
    // 'pszSource' is a C style string containing the shader's source code.
    // 'length' is the length of 'pszSource'.

    GLuint shader = glCreateShader(type);

    if (shader)
    {
        GLint compiled = 0;

        glShaderSource(shader, 1, &pszSource, &length);
        glCompileShader(shader);
        glGetShaderiv(shader, GL_COMPILE_STATUS, &compiled);

        if (!compiled)
        {
            GLsizei infoLogSize = 0;
            std::string infoLog;

            glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &infoLogSize);
            infoLog.resize(infoLogSize);
            glGetShaderInfoLog(shader, infoLogSize, &infoLogSize, &infoLog[0]);

            throw infoLog;
        }
    }

    return shader;
}

HWND CreateAppWindow(const WNDCLASSEX &wcl, const char *pszTitle)
{
    // Create a window that is centered on the desktop. It's exactly 1/4 the
    // size of the desktop. Don't allow it to be resized.

    DWORD wndExStyle = WS_EX_OVERLAPPEDWINDOW;
    DWORD wndStyle = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU |
                     WS_MINIMIZEBOX | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;

    HWND hWnd = CreateWindowEx(wndExStyle, wcl.lpszClassName, pszTitle,
                    wndStyle, 0, 0, 0, 0, 0, 0, wcl.hInstance, 0);

    if (hWnd)
    {
        int screenWidth = GetSystemMetrics(SM_CXSCREEN);
        int screenHeight = GetSystemMetrics(SM_CYSCREEN);
        int halfScreenWidth = screenWidth / 2;
        int halfScreenHeight = screenHeight / 2;
        int left = (screenWidth - halfScreenWidth) / 2;
        int top = (screenHeight - halfScreenHeight) / 2;
        RECT rc = {0};

        SetRect(&rc, left, top, left + halfScreenWidth, top + halfScreenHeight);
        AdjustWindowRectEx(&rc, wndStyle, FALSE, wndExStyle);
        MoveWindow(hWnd, rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top, TRUE);

        GetClientRect(hWnd, &rc);
        g_windowWidth = rc.right - rc.left;
        g_windowHeight = rc.bottom - rc.top;
    }

    return hWnd;
}

GLuint CreateNullTexture(int width, int height)
{
    // Create an empty white texture. This texture is applied to models
    // that don't have any texture maps. This trick allows the same shader to
    // be used to draw the model with and without textures applied.

    int pitch = ((width * 32 + 31) & ~31) >> 3; // align to 4-byte boundaries
    std::vector<GLubyte> pixels(pitch * height, 255);
    GLuint texture = 0;

    glGenTextures(1, &texture);
    glBindTexture(GL_TEXTURE_2D, texture);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, width, height, 0, GL_BGRA,
        GL_UNSIGNED_BYTE, &pixels[0]);

    return texture;
}

void EnableVerticalSync(bool enableVerticalSync)
{
    // WGL_EXT_swap_control.

    typedef BOOL (WINAPI * PFNWGLSWAPINTERVALEXTPROC)(GLint);

    static PFNWGLSWAPINTERVALEXTPROC wglSwapIntervalEXT =
        reinterpret_cast<PFNWGLSWAPINTERVALEXTPROC>(
        wglGetProcAddress("wglSwapIntervalEXT"));

    if (wglSwapIntervalEXT)
    {
        wglSwapIntervalEXT(enableVerticalSync ? 1 : 0);
        g_enableVerticalSync = enableVerticalSync;
    }
}

float GetElapsedTimeInSeconds()
{
    // Returns the elapsed time (in seconds) since the last time this function
    // was called. This elaborate setup is to guard against large spikes in
    // the time returned by QueryPerformanceCounter().

    static const int MAX_SAMPLE_COUNT = 50;

    static float frameTimes[MAX_SAMPLE_COUNT];
    static float timeScale = 0.0f;
    static float actualElapsedTimeSec = 0.0f;
    static INT64 freq = 0;
    static INT64 lastTime = 0;
    static int sampleCount = 0;
    static bool initialized = false;

    INT64 time = 0;
    float elapsedTimeSec = 0.0f;

    if (!initialized)
    {
        initialized = true;
        QueryPerformanceFrequency(reinterpret_cast<LARGE_INTEGER*>(&freq));
        QueryPerformanceCounter(reinterpret_cast<LARGE_INTEGER*>(&lastTime));
        timeScale = 1.0f / freq;
    }

    QueryPerformanceCounter(reinterpret_cast<LARGE_INTEGER*>(&time));
    elapsedTimeSec = (time - lastTime) * timeScale;
    lastTime = time;

    if (fabsf(elapsedTimeSec - actualElapsedTimeSec) < 1.0f)
    {
        memmove(&frameTimes[1], frameTimes, sizeof(frameTimes) - sizeof(frameTimes[0]));
        frameTimes[0] = elapsedTimeSec;

        if (sampleCount < MAX_SAMPLE_COUNT)
            ++sampleCount;
    }

    actualElapsedTimeSec = 0.0f;

    for (int i = 0; i < sampleCount; ++i)
        actualElapsedTimeSec += frameTimes[i];

    if (sampleCount > 0)
        actualElapsedTimeSec /= sampleCount;

    return actualElapsedTimeSec;
}

LPCSTR GetLightTypeStr()
{
    switch (g_lightType)
    {
    case DIR_LIGHT:
        return "Directional Lighting";

    case POINT_LIGHT:
        return "Point Lighting";

    case SPOT_LIGHT:
        return "Spot Lighting";

    default:
        return "Unknown Lighting";
    }
}

bool Init()
{
    try
    {
        InitGL();

        if (!OpenGLSupportsGLVersion(2, 0))
            throw std::runtime_error("This application requires OpenGL 2.0 support.");

        InitApp();
        return true;
    }
    catch (const std::exception &e)
    {
        std::ostringstream msg;

        msg << "Application initialization failed!" << std::endl << std::endl;
        msg << e.what();

        Log(msg.str().c_str());
        return false;
    }
}

void InitApp()
{
    // Setup fonts.

    if (!g_font.create("Arial", 10, GLFont::BOLD))
        throw std::runtime_error("Failed to create font.");

    // Load the GLSL parallax normal mapping shaders.

    std::string infoLog;

    if (!(g_dirParallaxNormalMappingShaderProgram = LoadShaderProgram(
            "Content/Shaders/parallax_normal_mapping_dir.glsl", infoLog)))
        throw std::runtime_error("Failed to load shader: parallax_normal_mapping_dir.glsl.\n" + infoLog);

    if (!(g_pointParallaxNormalMappingShaderProgram = LoadShaderProgram(
            "Content/Shaders/parallax_normal_mapping_point.glsl", infoLog)))
        throw std::runtime_error("Failed to load shader: parallax_normal_mapping_point.glsl.\n" + infoLog);

    if (!(g_spotParallaxNormalMappingShaderProgram = LoadShaderProgram(
            "Content/Shaders/parallax_normal_mapping_spot.glsl", infoLog)))
        throw std::runtime_error("Failed to load shader: parallax_normal_mapping_spot.glsl.\n" + infoLog);

    // Calculate the cos cutoff angles for use in the spot light shader.

    g_cosInnerCone = cosf(SPOT_INNER_CONE * (3.14159265358979323846f / 180.0f));
    g_cosOuterCone = cosf(SPOT_OUTER_CONE * (3.14159265358979323846f / 180.0f));

    // Load the textures.

    if (!(g_colorMapTexture = LoadTexture("Content/Textures/color_map.jpg")))
        throw std::runtime_error("Failed to load texture: color_map.jpg.");

    if (!(g_normalMapTexture = LoadTexture("Content/Textures/normal_map.jpg")))
        throw std::runtime_error("Failed to load texture: normal_map.jpg.");

    if (!(g_heightMapTexture = LoadTexture("Content/Textures/height_map.jpg")))
        throw std::runtime_error("Failed to load texture: height_map.jpg.");

    if (!(g_nullTexture = CreateNullTexture(2, 2)))
        throw std::runtime_error("Failed to create null texture.");

    // Initialize the cube model.

    InitCube();

    // Setup initial rendering states.

    glEnable(GL_TEXTURE_2D);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);

    UpdateLighting();
}

void InitCube()
{
    // Generate the tangent vectors for each face of the cube.

    Vertex *pVertex1 = 0;
    Vertex *pVertex2 = 0;
    Vertex *pVertex3 = 0;
    Vertex *pVertex4 = 0;
    float tangent[4] = {0.0f, 0.0f, 0.0f, 0.0f};

    for (int i = 0; i < 24; i += 4)
    {
        pVertex1 = &g_cube[i];
        pVertex2 = &g_cube[i + 1];
        pVertex3 = &g_cube[i + 2];
        pVertex4 = &g_cube[i + 3];

        CalcTangentVector(
            pVertex1->pos, pVertex2->pos, pVertex4->pos,
            pVertex1->texcoord, pVertex2->texcoord, pVertex4->texcoord,
            pVertex1->normal, tangent);

        pVertex1->tangent[0] = tangent[0];
        pVertex1->tangent[1] = tangent[1];
        pVertex1->tangent[2] = tangent[2];
        pVertex1->tangent[3] = tangent[3];

        pVertex2->tangent[0] = tangent[0];
        pVertex2->tangent[1] = tangent[1];
        pVertex2->tangent[2] = tangent[2];
        pVertex2->tangent[3] = tangent[3];

        pVertex3->tangent[0] = tangent[0];
        pVertex3->tangent[1] = tangent[1];
        pVertex3->tangent[2] = tangent[2];
        pVertex3->tangent[3] = tangent[3];

        pVertex4->tangent[0] = tangent[0];
        pVertex4->tangent[1] = tangent[1];
        pVertex4->tangent[2] = tangent[2];
        pVertex4->tangent[3] = tangent[3];
    }

    // Store the cube's geometry in a Vertex Buffer Object.

    glGenBuffers(1, &g_vertexBuffer);
    glBindBuffer(GL_ARRAY_BUFFER, g_vertexBuffer);
    glBufferData(GL_ARRAY_BUFFER, sizeof(g_cube), g_cube, GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void InitGL()
{
    if (!(g_hDC = GetDC(g_hWnd)))
        throw std::runtime_error("GetDC() failed.");

    int pf = 0;
    PIXELFORMATDESCRIPTOR pfd = {0};
    OSVERSIONINFO osvi = {0};

    pfd.nSize = sizeof(pfd);
    pfd.nVersion = 1;
    pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    pfd.iPixelType = PFD_TYPE_RGBA;
    pfd.cColorBits = 24;
    pfd.cDepthBits = 16;
    pfd.iLayerType = PFD_MAIN_PLANE;

    osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

    if (!GetVersionEx(&osvi))
        throw std::runtime_error("GetVersionEx() failed.");

    // When running under Windows Vista or later support desktop composition.
    if (osvi.dwMajorVersion > 6 || (osvi.dwMajorVersion == 6 && osvi.dwMinorVersion >= 0))
        pfd.dwFlags |=  PFD_SUPPORT_COMPOSITION;

    ChooseBestMultiSampleAntiAliasingPixelFormat(pf, g_msaaSamples);

    if (!pf)
        pf = ChoosePixelFormat(g_hDC, &pfd);

    if (!SetPixelFormat(g_hDC, pf, &pfd))
        throw std::runtime_error("SetPixelFormat() failed.");

    if (!(g_hRC = wglCreateContext(g_hDC)))
        throw std::runtime_error("wglCreateContext() failed.");

    if (!wglMakeCurrent(g_hDC, g_hRC))
        throw std::runtime_error("wglMakeCurrent() failed.");

    EnableVerticalSync(false);

    // Check for GL_EXT_texture_filter_anisotropic support.
    if (OpenGLExtensionSupported("GL_EXT_texture_filter_anisotropic"))
        glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &g_maxAnisotrophy);
    else
        g_maxAnisotrophy = 1;
}

GLuint LinkShaders(GLuint vertShader, GLuint fragShader)
{
    // Links the compiled vertex and/or fragment shaders into an executable
    // shader program. Returns the executable shader object. If the shaders
    // failed to link into an executable shader program, then a std::string
    // object is thrown containing the info log.

    GLuint program = glCreateProgram();

    if (program)
    {
        GLint linked = 0;

        if (vertShader)
            glAttachShader(program, vertShader);

        if (fragShader)
            glAttachShader(program, fragShader);

        glLinkProgram(program);
        glGetProgramiv(program, GL_LINK_STATUS, &linked);

        if (!linked)
        {
            GLsizei infoLogSize = 0;
            std::string infoLog;

            glGetProgramiv(program, GL_INFO_LOG_LENGTH, &infoLogSize);
            infoLog.resize(infoLogSize);
            glGetProgramInfoLog(program, infoLogSize, &infoLogSize, &infoLog[0]);

            throw infoLog;
        }

        // Mark the two attached shaders for deletion. These two shaders aren't
        // deleted right now because both are already attached to a shader
        // program. When the shader program is deleted these two shaders will
        // be automatically detached and deleted.

        if (vertShader)
            glDeleteShader(vertShader);

        if (fragShader)
            glDeleteShader(fragShader);
    }

    return program;
}

GLuint LoadShaderProgram(const char *pszFilename, std::string &infoLog)
{
    infoLog.clear();

    GLuint program = 0;
    std::string buffer;

    // Read the text file containing the GLSL shader program.
    // This file contains 1 vertex shader and 1 fragment shader.
    ReadTextFile(pszFilename, buffer);

    // Compile and link the vertex and fragment shaders.
    if (buffer.length() > 0)
    {
        const GLchar *pSource = 0;
        GLint length = 0;
        GLuint vertShader = 0;
        GLuint fragShader = 0;

        std::string::size_type vertOffset = buffer.find("[vert]");
        std::string::size_type fragOffset = buffer.find("[frag]");

        try
        {
            // Get the vertex shader source and compile it.
            // The source is between the [vert] and [frag] tags.
            if (vertOffset != std::string::npos)
            {
                vertOffset += 6;        // skip over the [vert] tag
                pSource = reinterpret_cast<const GLchar *>(&buffer[vertOffset]);
                length = static_cast<GLint>(fragOffset - vertOffset);
                vertShader = CompileShader(GL_VERTEX_SHADER, pSource, length);
            }

            // Get the fragment shader source and compile it.
            // The source is between the [frag] tag and the end of the file.
            if (fragOffset != std::string::npos)
            {
                fragOffset += 6;        // skip over the [frag] tag
                pSource = reinterpret_cast<const GLchar *>(&buffer[fragOffset]);
                length = static_cast<GLint>(buffer.length() - fragOffset - 1);
                fragShader = CompileShader(GL_FRAGMENT_SHADER, pSource, length);
            }

            // Now link the vertex and fragment shaders into a shader program.
            program = LinkShaders(vertShader, fragShader);
        }
        catch (const std::string &errors)
        {
            infoLog = errors;
        }
    }

    return program;
}

GLuint LoadTexture(const char *pszFilename)
{
    return LoadTexture(pszFilename, GL_LINEAR, GL_LINEAR_MIPMAP_LINEAR,
        GL_REPEAT, GL_REPEAT);
}

GLuint LoadTexture(const char *pszFilename, GLint magFilter, GLint minFilter,
                   GLint wrapS, GLint wrapT)
{
    GLuint id = 0;
    Bitmap bitmap;

    if (bitmap.loadPicture(pszFilename))
    {
        // The Bitmap class loads images and orients them top-down.
        // OpenGL expects bitmap images to be oriented bottom-up.
        bitmap.flipVertical();

        glGenTextures(1, &id);
        glBindTexture(GL_TEXTURE_2D, id);

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, magFilter);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, minFilter);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, wrapS);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, wrapT);
        glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

        if (g_maxAnisotrophy > 1)
        {
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT,
                g_maxAnisotrophy);
        }

        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, bitmap.width, bitmap.height,
            0, GL_BGRA, GL_UNSIGNED_BYTE, bitmap.getPixels());

        glBindTexture(GL_TEXTURE_2D, 0);
    }

    return id;
}

void Log(const char *pszMessage)
{
    MessageBox(0, pszMessage, "Error", MB_ICONSTOP);
}

void ProcessMouseInput(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    // Use the left mouse button to track the camera.
    // Use the middle mouse button to dolly the camera.
    // Use the right mouse button to orbit the camera.

    enum CameraMode {CAMERA_NONE, CAMERA_TRACK, CAMERA_DOLLY, CAMERA_ORBIT};

    static CameraMode cameraMode = CAMERA_NONE;
    static POINT ptMousePrev = {0};
    static POINT ptMouseCurrent = {0};
    static int mouseButtonsDown = 0;
    static float dx = 0.0f;
    static float dy = 0.0f;

    switch (msg)
    {
    case WM_LBUTTONDOWN:
        cameraMode = CAMERA_TRACK;
        ++mouseButtonsDown;
        SetCapture(hWnd);
        ptMousePrev.x = static_cast<int>(static_cast<short>(LOWORD(lParam)));
        ptMousePrev.y = static_cast<int>(static_cast<short>(HIWORD(lParam)));
        ClientToScreen(hWnd, &ptMousePrev);
        break;

    case WM_RBUTTONDOWN:
        cameraMode = CAMERA_ORBIT;
        ++mouseButtonsDown;
        SetCapture(hWnd);
        ptMousePrev.x = static_cast<int>(static_cast<short>(LOWORD(lParam)));
        ptMousePrev.y = static_cast<int>(static_cast<short>(HIWORD(lParam)));
        ClientToScreen(hWnd, &ptMousePrev);
        break;

    case WM_MBUTTONDOWN:
        cameraMode = CAMERA_DOLLY;
        ++mouseButtonsDown;
        SetCapture(hWnd);
        ptMousePrev.x = static_cast<int>(static_cast<short>(LOWORD(lParam)));
        ptMousePrev.y = static_cast<int>(static_cast<short>(HIWORD(lParam)));
        ClientToScreen(hWnd, &ptMousePrev);
        break;

    case WM_MOUSEMOVE:
        ptMouseCurrent.x = static_cast<int>(static_cast<short>(LOWORD(lParam)));
        ptMouseCurrent.y = static_cast<int>(static_cast<short>(HIWORD(lParam)));
        ClientToScreen(hWnd, &ptMouseCurrent);

        switch (cameraMode)
        {
        case CAMERA_TRACK:
            dx = static_cast<float>(ptMouseCurrent.x - ptMousePrev.x);
            dx *= MOUSE_TRACK_SPEED;

            dy = static_cast<float>(ptMouseCurrent.y - ptMousePrev.y);
            dy *= MOUSE_TRACK_SPEED;

            g_eyePos[0] -= dx;
            g_eyePos[1] += dy;

            g_targetPos[0] -= dx;
            g_targetPos[1] += dy;

            UpdateLighting();
            break;

        case CAMERA_DOLLY:
            dy = static_cast<float>(ptMouseCurrent.y - ptMousePrev.y);
            dy *= MOUSE_DOLLY_SPEED;

            g_eyePos[2] -= dy;

            if (g_eyePos[2] < DOLLY_MIN)
                g_eyePos[2] = DOLLY_MIN;

            if (g_eyePos[2] > DOLLY_MAX)
                g_eyePos[2] = DOLLY_MAX;

            UpdateLighting();
            break;

        case CAMERA_ORBIT:
            dx = static_cast<float>(ptMouseCurrent.x - ptMousePrev.x);
            dx *= MOUSE_ORBIT_SPEED;

            dy = static_cast<float>(ptMouseCurrent.y - ptMousePrev.y);
            dy *= MOUSE_ORBIT_SPEED;

            g_heading += dx;
            g_pitch += dy;

            if (g_pitch > 90.0f)
                g_pitch = 90.0f;

            if (g_pitch < -90.0f)
                g_pitch = -90.0f;

            break;
        }

        ptMousePrev.x = ptMouseCurrent.x;
        ptMousePrev.y = ptMouseCurrent.y;
        break;

    case WM_LBUTTONUP:
    case WM_RBUTTONUP:
    case WM_MBUTTONUP:
        if (--mouseButtonsDown <= 0)
        {
            mouseButtonsDown = 0;
            cameraMode = CAMERA_NONE;
            ReleaseCapture();
        }
        else
        {
            if (wParam & MK_LBUTTON)
                cameraMode = CAMERA_TRACK;
            else if (wParam & MK_RBUTTON)
                cameraMode = CAMERA_ORBIT;
            else if (wParam & MK_MBUTTON)
                cameraMode = CAMERA_DOLLY;
        }
        break;

    default:
        break;
    }
}

void ReadTextFile(const char *pszFilename, std::string &buffer)
{
    std::ifstream file(pszFilename, std::ios::binary);

    if (file.is_open())
    {
        file.seekg(0, std::ios::end);

        std::ifstream::pos_type fileSize = file.tellg();

        buffer.resize(static_cast<unsigned int>(fileSize));
        file.seekg(0, std::ios::beg);
        file.read(&buffer[0], fileSize);
    }
}

void RenderCube()
{
    glActiveTexture(GL_TEXTURE2);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, g_heightMapTexture);

    glActiveTexture(GL_TEXTURE1);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, g_normalMapTexture);

    glActiveTexture(GL_TEXTURE0);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, (g_disableColorMapTexture) ? g_nullTexture : g_colorMapTexture);   

    glBindBuffer(GL_ARRAY_BUFFER, g_vertexBuffer);

    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, sizeof(Vertex), BUFFER_OFFSET(0));

    glClientActiveTexture(GL_TEXTURE0);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glTexCoordPointer(2, GL_FLOAT, sizeof(Vertex), BUFFER_OFFSET(sizeof(float) * 3));

    glEnableClientState(GL_NORMAL_ARRAY);
    glNormalPointer(GL_FLOAT, sizeof(Vertex), BUFFER_OFFSET(sizeof(float) * 5));

    glClientActiveTexture(GL_TEXTURE1);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glTexCoordPointer(4, GL_FLOAT, sizeof(Vertex), BUFFER_OFFSET(sizeof(float) * 8));

    glDrawArrays(GL_QUADS, 0, sizeof(g_cube) / sizeof(g_cube[0]));

    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);
    glClientActiveTexture(GL_TEXTURE0);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);

    glBindBuffer(GL_ARRAY_BUFFER, 0);

    glActiveTexture(GL_TEXTURE2);
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisable(GL_TEXTURE_2D);

    glActiveTexture(GL_TEXTURE1);
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisable(GL_TEXTURE_2D);

    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisable(GL_TEXTURE_2D);
}

void RenderFrame()
{
    // Setup view port.

    glViewport(0, 0, g_windowWidth, g_windowHeight);
    glClearColor(0.3f, 0.5f, 0.9f, 0.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    // Setup perspective projection matrix.

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(CAMERA_FOVY,
        static_cast<float>(g_windowWidth) / static_cast<float>(g_windowHeight),
        CAMERA_ZNEAR, CAMERA_ZFAR);

    // Setup view matrix.

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    gluLookAt(g_eyePos[0], g_eyePos[1], g_eyePos[2],
        g_targetPos[0], g_targetPos[1], g_targetPos[2],
        0.0f, 1.0f, 0.0f);

    // Setup lighting and shaders.

    glPushAttrib(GL_LIGHTING_BIT);

    switch (g_lightType)
    {
    case DIR_LIGHT:
        glLightfv(GL_LIGHT0, GL_POSITION, g_lightPos);
        glUseProgram(g_dirParallaxNormalMappingShaderProgram);
        glUniform1i(glGetUniformLocation(g_dirParallaxNormalMappingShaderProgram, "colorMap"), 0);
        glUniform1i(glGetUniformLocation(g_dirParallaxNormalMappingShaderProgram, "normalMap"), 1);
        glUniform1i(glGetUniformLocation(g_dirParallaxNormalMappingShaderProgram, "heightMap"), 2);
        glUniform1f(glGetUniformLocation(g_dirParallaxNormalMappingShaderProgram, "scale"), g_scaleBias[0]);
        glUniform1f(glGetUniformLocation(g_dirParallaxNormalMappingShaderProgram, "bias"), g_scaleBias[1]);
        glUniform1i(glGetUniformLocation(g_dirParallaxNormalMappingShaderProgram, "enableParallax"), !g_disableParallax);
        break;

    case POINT_LIGHT:
        glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 180.0f);
        glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, 0.0f);
        glLightfv(GL_LIGHT0, GL_POSITION, g_lightPos);
        glUseProgram(g_pointParallaxNormalMappingShaderProgram);
        glUniform1i(glGetUniformLocation(g_pointParallaxNormalMappingShaderProgram, "colorMap"), 0);
        glUniform1i(glGetUniformLocation(g_pointParallaxNormalMappingShaderProgram, "normalMap"), 1);
        glUniform1i(glGetUniformLocation(g_pointParallaxNormalMappingShaderProgram, "heightMap"), 2);
        glUniform1f(glGetUniformLocation(g_pointParallaxNormalMappingShaderProgram, "lightRadius"), LIGHT_RADIUS);
        glUniform1f(glGetUniformLocation(g_pointParallaxNormalMappingShaderProgram, "scale"), g_scaleBias[0]);
        glUniform1f(glGetUniformLocation(g_pointParallaxNormalMappingShaderProgram, "bias"), g_scaleBias[1]);
        glUniform1i(glGetUniformLocation(g_pointParallaxNormalMappingShaderProgram, "enableParallax"), !g_disableParallax);
        break;

    case SPOT_LIGHT:
        glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, SPOT_OUTER_CONE);
        glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, 2.0f);
        glLightfv(GL_LIGHT0, GL_POSITION, g_lightPos);
        glUseProgram(g_spotParallaxNormalMappingShaderProgram);
        glUniform1i(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "colorMap"), 0);
        glUniform1i(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "normalMap"), 1);
        glUniform1i(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "heightMap"), 2);
        glUniform1f(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "cosInnerCone"), g_cosInnerCone);
        glUniform1f(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "cosOuterCone"), g_cosOuterCone);
        glUniform1f(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "lightRadius"), LIGHT_RADIUS);
        glUniform1f(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "scale"), g_scaleBias[0]);
        glUniform1f(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "bias"), g_scaleBias[1]);
        glUniform1i(glGetUniformLocation(g_spotParallaxNormalMappingShaderProgram, "enableParallax"), !g_disableParallax);
        break;
    }

    // Render the cube.

    glRotatef(g_pitch, 1.0f, 0.0f, 0.0f);
    glRotatef(g_heading, 0.0f, 1.0f, 0.0f);

    RenderCube();

    glUseProgram(0);

    glPopAttrib();

    // Render the diagnostic text. This is done using the fixed function
    // pipeline so it's important that any shaders be disabled prior to
    // calling this function.

    RenderText();
}

void RenderText()
{
    std::ostringstream output;

    if (g_displayHelp)
    {
        output
            << "Left mouse click and drag to track camera" << std::endl
            << "Middle mouse click and drag to dolly camera" << std::endl
            << "Right mouse click and drag to orbit camera" << std::endl
            << std::endl
            << "Press SPACE to cycle forwards through the light types" << std::endl
            << "Press BACKSPACE to cycle backwards through the light types" << std::endl
            << "Press B to increase the parallax height bias" << std::endl
            << "Press b to decrease the parallax height bias" << std::endl
            << "Press P to toggle between parallax normal mapping and normal mapping" << std::endl
            << "Press S to increase the parallax height scale" << std::endl
            << "Press s to decrease the parallax height scale" << std::endl
            << "Press T to enable/disable textures" << std::endl
            << "Press V to enable/disable vertical sync" << std::endl
            << "Press ALT + ENTER to toggle full screen" << std::endl
            << "Press ESC to exit" << std::endl
            << std::endl
            << "Press H to hide help";
    }
    else
    {
        output
            << "FPS: " << g_framesPerSecond << std::endl
            << "Multisample anti-aliasing: " << g_msaaSamples << "x" << std::endl
            << "Anisotropic filtering: " << g_maxAnisotrophy << "x" << std::endl
            << "Vertical sync: " << (g_enableVerticalSync ? "enabled" : "disabled") << std::endl
            << "Lighting: " << GetLightTypeStr() << std::endl;
        
        if (g_disableParallax)
        {
            output << "Technique: Normal mapping" << std::endl;
        }
        else
        {
            output
                << "Technique: Parallax normal mapping" << std::endl
                << "Parallax scale: " << g_scaleBias[0] << std::endl
                << "Parallax bias: " << g_scaleBias[1] << std::endl;
        }
            
        output
            << std::endl
            << "Press H to display help";
    }

    g_font.begin();
    g_font.setColor(1.0f, 1.0f, 0.0f);
    g_font.drawText(1, 1, output.str().c_str());
    g_font.end();
}

void SetProcessorAffinity()
{
    // Assign the current thread to one processor. This ensures that timing
    // code runs on only one processor, and will not suffer any ill effects
    // from power management.
    //
    // Based on DXUTSetProcessorAffinity() function from the DXUT framework.

    DWORD_PTR dwProcessAffinityMask = 0;
    DWORD_PTR dwSystemAffinityMask = 0;
    HANDLE hCurrentProcess = GetCurrentProcess();

    if (!GetProcessAffinityMask(hCurrentProcess, &dwProcessAffinityMask, &dwSystemAffinityMask))
        return;

    if (dwProcessAffinityMask)
    {
        // Find the lowest processor that our process is allowed to run against.

        DWORD_PTR dwAffinityMask = (dwProcessAffinityMask & ((~dwProcessAffinityMask) + 1));

        // Set this as the processor that our thread must always run against.
        // This must be a subset of the process affinity mask.

        HANDLE hCurrentThread = GetCurrentThread();

        if (hCurrentThread != INVALID_HANDLE_VALUE)
        {
            SetThreadAffinityMask(hCurrentThread, dwAffinityMask);
            CloseHandle(hCurrentThread);
        }
    }

    CloseHandle(hCurrentProcess);
}

void ToggleFullScreen()
{
    static DWORD savedExStyle;
    static DWORD savedStyle;
    static RECT rcSaved;

    g_isFullScreen = !g_isFullScreen;

    if (g_isFullScreen)
    {
        // Moving to full screen mode.

        savedExStyle = GetWindowLong(g_hWnd, GWL_EXSTYLE);
        savedStyle = GetWindowLong(g_hWnd, GWL_STYLE);
        GetWindowRect(g_hWnd, &rcSaved);

        SetWindowLong(g_hWnd, GWL_EXSTYLE, 0);
        SetWindowLong(g_hWnd, GWL_STYLE, WS_POPUP | WS_CLIPCHILDREN | WS_CLIPSIBLINGS);
        SetWindowPos(g_hWnd, HWND_TOPMOST, 0, 0, 0, 0,
            SWP_NOMOVE | SWP_NOSIZE | SWP_FRAMECHANGED | SWP_SHOWWINDOW);

        g_windowWidth = GetSystemMetrics(SM_CXSCREEN);
        g_windowHeight = GetSystemMetrics(SM_CYSCREEN);

        SetWindowPos(g_hWnd, HWND_TOPMOST, 0, 0,
            g_windowWidth, g_windowHeight, SWP_SHOWWINDOW);
    }
    else
    {
        // Moving back to windowed mode.

        SetWindowLong(g_hWnd, GWL_EXSTYLE, savedExStyle);
        SetWindowLong(g_hWnd, GWL_STYLE, savedStyle);
        SetWindowPos(g_hWnd, HWND_NOTOPMOST, 0, 0, 0, 0,
            SWP_NOMOVE | SWP_NOSIZE | SWP_FRAMECHANGED | SWP_SHOWWINDOW);

        g_windowWidth = rcSaved.right - rcSaved.left;
        g_windowHeight = rcSaved.bottom - rcSaved.top;

        SetWindowPos(g_hWnd, HWND_NOTOPMOST, rcSaved.left, rcSaved.top,
            g_windowWidth, g_windowHeight, SWP_SHOWWINDOW);
    }
}

void UpdateFrame(float elapsedTimeSec)
{
    UpdateFrameRate(elapsedTimeSec);
}

void UpdateFrameRate(float elapsedTimeSec)
{
    static float accumTimeSec = 0.0f;
    static int frames = 0;

    accumTimeSec += elapsedTimeSec;

    if (accumTimeSec > 1.0f)
    {
        g_framesPerSecond = frames;

        frames = 0;
        accumTimeSec = 0.0f;
    }
    else
    {
        ++frames;
    }
}

void UpdateLighting()
{
    switch (g_lightType)
    {
    case DIR_LIGHT:
        g_lightPos[0] = 0.0f;
        g_lightPos[1] = 0.0f;
        g_lightPos[2] = 1.0f;
        g_lightPos[3] = 0.0f;
        break;

    case POINT_LIGHT:
    case SPOT_LIGHT:
        g_lightPos[0] = g_eyePos[0];
        g_lightPos[1] = g_eyePos[1];
        g_lightPos[2] = g_eyePos[2];
        g_lightPos[3] = g_eyePos[3];
        break;
    }
}