//-----------------------------------------------------------------------------
// Copyright (c) 2008-2011 dhpoware. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//-----------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;

namespace XNARenderToTexture
{
    /// <summary>
    /// This Windows XNA application demonstrates rendering to a texture.
    /// Rendering to a texture was fairly easy to do in XNA 3. With XNA 4
    /// rendering to a texture is even easier to do! XNA 3 and earlier
    /// required the developer to manually manage both the RenderTarget2D
    /// object and its corresponding DepthStencilBuffer object. This proved
    /// to be error prone. With XNA 4 the DepthStencilBuffer class has been
    /// removed from the XNA 4 Framework. The RenderTarger2D class now manages
    /// the depth-stencil buffer. This demo will show you how much easier
    /// rendering to a texture has become in XNA 4.
    /// </summary>
    public class Demo : Microsoft.Xna.Framework.Game
    {
        private static void Main()
        {
            using (Demo demo = new Demo())
            {
                demo.Run();
            }
        }

        private const float BILLBOARD_MAX_ROTATION = 80.0f;
        private const float BILLBOARD_MIN_ROTATION = -80.0f;
        private const float MOUSE_ROTATION_SPEED = 0.30f;
        private const int RENDERTARGET_WIDTH = 1024;
        private const int RENDERTARGET_HEIGHT = 1024;

        private GraphicsDeviceManager graphics;
        private SpriteBatch spriteBatch;
        private SpriteFont spriteFont;
        private Effect billboardEffect;
        private RenderTarget2D renderTarget;
        private VertexBuffer vertexBuffer;
        private Model model;
        private Matrix[] modelTransforms;
        private Vector2 modelRotation;
        private Vector2 billboardRotation;
        private Vector2 fontPos;
        private Vector3 cameraPos;
        private KeyboardState currentKeyboardState;
        private KeyboardState prevKeyboardState;
        private MouseState currentMouseState;
        private MouseState prevMouseState;
        private int windowWidth;
        private int windowHeight;
        private int frames;
        private int framesPerSecond;
        private TimeSpan elapsedTime = TimeSpan.Zero;
        private bool displayHelp;

        public Demo()
        {
            graphics = new GraphicsDeviceManager(this);
            Content.RootDirectory = "Content";

            Window.Title = "XNA 4.0 Render to Texture Demo";
            IsMouseVisible = true;
            IsFixedTimeStep = false;
        }

        protected override void Initialize()
        {
            // Setup the window to be a quarter the size of the desktop.
            
            windowWidth = GraphicsDevice.DisplayMode.Width / 2;
            windowHeight = GraphicsDevice.DisplayMode.Height / 2;

            // Setup frame buffer.
            
            graphics.SynchronizeWithVerticalRetrace = false;
            graphics.PreferredBackBufferWidth = windowWidth;
            graphics.PreferredBackBufferHeight = windowHeight;
            graphics.PreferMultiSampling = true;
            graphics.ApplyChanges();

            // Position the text.
            
            fontPos = new Vector2(1.0f, 1.0f);

            // Position the camera.
            
            cameraPos = new Vector3(0.0f, 0.0f, 4.0f);

            // Setup the initial input states.
            
            currentKeyboardState = Keyboard.GetState();
            currentMouseState = Mouse.GetState();

            // Create a quad for the billboard.

            float w = windowWidth / windowHeight;
            float h = 1.0f;

            VertexPositionTexture[] vertices =
            {
                new VertexPositionTexture(new Vector3(-w,  h, 0.0f), new Vector2(0.0f, 0.0f)),
                new VertexPositionTexture(new Vector3( w,  h, 0.0f), new Vector2(1.0f, 0.0f)),
                new VertexPositionTexture(new Vector3(-w, -h, 0.0f), new Vector2(0.0f, 1.0f)),
                new VertexPositionTexture(new Vector3( w, -h, 0.0f), new Vector2(1.0f, 1.0f))
            };

            vertexBuffer = new VertexBuffer(GraphicsDevice, typeof(VertexPositionTexture), vertices.Length, BufferUsage.WriteOnly);
            vertexBuffer.SetData(vertices);
            
            // The model will be drawn in this render target.
            
            renderTarget = new RenderTarget2D(GraphicsDevice, RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, true, SurfaceFormat.Color, DepthFormat.Depth24Stencil8);

            base.Initialize();
        }

        protected override void LoadContent()
        {
            spriteBatch = new SpriteBatch(GraphicsDevice);
            spriteFont = Content.Load<SpriteFont>(@"Fonts\DemoFont");

            billboardEffect = Content.Load<Effect>(@"Effects\billboard");
            
            model = Content.Load<Model>(@"Models\teapot");
            modelTransforms = new Matrix[model.Bones.Count];
            model.CopyAbsoluteBoneTransformsTo(modelTransforms);
        }

        protected override void UnloadContent()
        {
        }

        private bool KeyJustPressed(Keys key)
        {
            return currentKeyboardState.IsKeyDown(key) && prevKeyboardState.IsKeyUp(key);
        }

        private void ProcessKeyboard()
        {
            prevKeyboardState = currentKeyboardState;
            currentKeyboardState = Keyboard.GetState();

            if (KeyJustPressed(Keys.Escape))
                this.Exit();

            if (KeyJustPressed(Keys.H))
                displayHelp = !displayHelp;

            if (currentKeyboardState.IsKeyDown(Keys.LeftAlt) ||
                currentKeyboardState.IsKeyDown(Keys.RightAlt))
            {
                if (KeyJustPressed(Keys.Enter))
                    ToggleFullScreen();
            }
        }

        private void ProcessMouse()
        {
            prevMouseState = currentMouseState;
            currentMouseState = Mouse.GetState();

            float heading = currentMouseState.X - prevMouseState.X;
            float pitch = currentMouseState.Y - prevMouseState.Y;

            heading *= MOUSE_ROTATION_SPEED;
            pitch *= MOUSE_ROTATION_SPEED;

            if (currentMouseState.LeftButton == ButtonState.Pressed)
            {
                billboardRotation.X += pitch;
                billboardRotation.Y += heading;

                if (billboardRotation.X > BILLBOARD_MAX_ROTATION)
                    billboardRotation.X = BILLBOARD_MAX_ROTATION;

                if (billboardRotation.X < BILLBOARD_MIN_ROTATION)
                    billboardRotation.X = BILLBOARD_MIN_ROTATION;

                if (billboardRotation.Y > BILLBOARD_MAX_ROTATION)
                    billboardRotation.Y = BILLBOARD_MAX_ROTATION;

                if (billboardRotation.Y < BILLBOARD_MIN_ROTATION)
                    billboardRotation.Y = BILLBOARD_MIN_ROTATION;
            }
            else if (currentMouseState.RightButton == ButtonState.Pressed)
            {
                modelRotation.X += pitch;
                modelRotation.Y += heading;
            }
        }

        private void ToggleFullScreen()
        {
            int newWidth = 0;
            int newHeight = 0;

            graphics.IsFullScreen = !graphics.IsFullScreen;

            if (graphics.IsFullScreen)
            {
                newWidth = GraphicsDevice.DisplayMode.Width;
                newHeight = GraphicsDevice.DisplayMode.Height;
            }
            else
            {
                newWidth = windowWidth;
                newHeight = windowHeight;
            }

            graphics.PreferredBackBufferWidth = newWidth;
            graphics.PreferredBackBufferHeight = newHeight;
            graphics.ApplyChanges();
        }

        protected override void Update(GameTime gameTime)
        {
            if (!IsActive)
                return;

            ProcessKeyboard();
            ProcessMouse();
            UpdateFrameRate(gameTime);

            base.Update(gameTime);
        }

        private void UpdateFrameRate(GameTime gameTime)
        {
            elapsedTime += gameTime.ElapsedGameTime;

            if (elapsedTime > TimeSpan.FromSeconds(1))
            {
                elapsedTime -= TimeSpan.FromSeconds(1);
                framesPerSecond = frames;
                frames = 0;
            }
        }

        private void IncrementFrameCounter()
        {
            ++frames;
        }

        private void DrawModel()
        {
            Matrix proj;
            Matrix view;
            Matrix world;

            proj = Matrix.CreatePerspectiveFieldOfView(MathHelper.PiOver4,
                    (float)renderTarget.Width / (float)renderTarget.Height,
                    0.1f, 100.0f);
            
            view = Matrix.CreateLookAt(cameraPos, Vector3.Zero, Vector3.Up);
            
            world = Matrix.CreateRotationY(MathHelper.ToRadians(modelRotation.Y)) *
                    Matrix.CreateRotationX(MathHelper.ToRadians(modelRotation.X));
                       
            foreach (ModelMesh mesh in model.Meshes)
            {
                foreach (BasicEffect effect in mesh.Effects)
                {
                    effect.PreferPerPixelLighting = true;
                    effect.EnableDefaultLighting();
                    effect.View = view;
                    effect.Projection = proj;
                    effect.World = modelTransforms[mesh.ParentBone.Index] * world;
                }

                mesh.Draw();
            }
        }

        private void DrawBillboard()
        {
            Matrix proj;
            Matrix view;
            Matrix world;
            Matrix worldViewProj;
            
            proj = Matrix.CreatePerspectiveFieldOfView(MathHelper.PiOver4,
                    (float)windowWidth / (float)windowHeight, 0.1f, 100.0f);

            view = Matrix.CreateLookAt(cameraPos, Vector3.Zero, Vector3.Up);

            world = Matrix.CreateRotationY(MathHelper.ToRadians(billboardRotation.Y)) *
                    Matrix.CreateRotationX(MathHelper.ToRadians(billboardRotation.X));

            worldViewProj = world * view * proj;

            billboardEffect.Parameters["worldViewProjectionMatrix"].SetValue(worldViewProj);
            billboardEffect.Parameters["colorMapTexture"].SetValue((Texture2D)renderTarget);
            billboardEffect.CurrentTechnique = billboardEffect.Techniques["Billboard"];

            GraphicsDevice.SetVertexBuffer(vertexBuffer);

            foreach (EffectPass pass in billboardEffect.CurrentTechnique.Passes)
            {
                pass.Apply();
                GraphicsDevice.DrawPrimitives(PrimitiveType.TriangleStrip, 0, 2);
            }
        }

        private void DrawText()
        {
            StringBuilder buffer = new StringBuilder();

            if (displayHelp)
            {
                buffer.AppendLine("Left mouse click and drag to rotate billboard");
                buffer.AppendLine("Right mouse click and drag to rotate teapot");
                buffer.AppendLine();
                buffer.AppendLine("Press ALT and ENTER to toggle full screen");
                buffer.AppendLine("Press ESCAPE to exit");
                buffer.AppendLine();
                buffer.AppendLine("Press H to hide help");
            }
            else
            {
                buffer.AppendFormat("FPS: {0}\n", framesPerSecond);
                                                
                buffer.AppendLine();
                buffer.AppendLine("Press H to display help");
            }

            spriteBatch.Begin(SpriteSortMode.Deferred, BlendState.AlphaBlend);
            spriteBatch.DrawString(spriteFont, buffer.ToString(), fontPos, Color.Yellow);
            spriteBatch.End();
        }

        protected override void Draw(GameTime gameTime)
        {
            if (!IsActive)
                return;

            // Draw the model in the render target.

            GraphicsDevice.SetRenderTarget(renderTarget);
            GraphicsDevice.Clear(Color.Black);
            GraphicsDevice.BlendState = BlendState.Opaque;
            GraphicsDevice.DepthStencilState = DepthStencilState.Default;
            GraphicsDevice.SamplerStates[0] = SamplerState.LinearWrap;

            DrawModel();
            
            // Now draw the textured quad (billboard) and use the render
            // target as the color map texture that will be passed to the
            // Billboard.fx Effect.

            GraphicsDevice.SetRenderTarget(null);
            GraphicsDevice.Clear(Color.CornflowerBlue);
            GraphicsDevice.BlendState = BlendState.Opaque;
            GraphicsDevice.DepthStencilState = DepthStencilState.Default;
            GraphicsDevice.SamplerStates[0] = SamplerState.LinearWrap;

            DrawBillboard();
            
            DrawText();
            base.Draw(gameTime);
            IncrementFrameCounter();
        }
    }
}